/**
  ******************************************************************************
  * @file    py32f005_ll_opa.h
  * @author  MCU Application Team
  * @brief   Header file of OPA LL module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __PY32F005_LL_OPA_H
#define __PY32F005_LL_OPA_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32f0xx.h"

/** @addtogroup PY32F005_LL_Driver
  * @{
  */

#if defined(OPA1) || defined(OPA)

/** @defgroup OPA_LL OPA
  * @{
  */

/* Private types -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

/* Private constants ---------------------------------------------------------*/
/** @defgroup OPA_LL_Private_Constants OPA Private Constants
  * @{
  */

/**
  * @}
  */

/* Private macros ------------------------------------------------------------*/
/** @defgroup OPA_LL_Private_Macros OPA Private Macros
  * @{
  */
  
/**
  * @}
  */


/* Exported types ------------------------------------------------------------*/
#if defined(USE_FULL_LL_DRIVER)


#endif /* USE_FULL_LL_DRIVER */

/* Exported constants --------------------------------------------------------*/
/** @defgroup OPA_LL_Exported_Constants OPA Exported Constants
  * @{
  */

/** @defgroup OPA_LL_EC_FUNCTIONAL_MODE OPA functional mode
  * @{
  */
#define LL_OPA_MODE_STANDALONE        (0x00000000UL)                                                /*!< OPA functional mode, OPA operation in standalone */
#define LL_OPA_MODE_PGA               (OPA_OCR_OPA_VMSEL_0)                                         /*!< OPA functional mode, OPA operation in PGA */
#define LL_OPA_MODE_FOLLOWER          (OPA_OCR_OPA_VMSEL_1)                                         /*!< OPA functional mode, OPA operation in follower */

#define LL_OPA_MODE_PGA_IO0_BIAS      (OPA_OCR_OPA_PGA_GAIN_3 | OPA_OCR_OPA_VMSEL_0)                /*!< In PGA mode, the inverting input is connected to VINM0 */
 

/**
  * @}
  */

/** @defgroup OPA_LL_EC_MODE_PGA_GAIN OPA PGA gain (relevant when OPA is in functional mode PGA)
  * @{
  */
#define LL_OPA_PGA_GAIN_2_OR_MINUS_1     (0x00000000UL)                                                               /*!< OPA PGA gain 2 or -1 */
#define LL_OPA_PGA_GAIN_4_OR_MINUS_3     (                                                  OPA_OCR_OPA_PGA_GAIN_0)   /*!< OPA PGA gain 4 or -3  */
#define LL_OPA_PGA_GAIN_8_OR_MINUS_7     (                         OPA_OCR_OPA_PGA_GAIN_1                         )   /*!< OPA PGA gain 8 or -7  */
#define LL_OPA_PGA_GAIN_16_OR_MINUS_15   (                         OPA_OCR_OPA_PGA_GAIN_1 | OPA_OCR_OPA_PGA_GAIN_0)   /*!< OPA PGA gain 16 or -15 */
/* #define LL_OPA_PGA_GAIN_32_OR_MINUS_31   (OPA_OCR_OPA_PGA_GAIN_2                                                  ) */   /*!< OPA PGA gain 32 or -31 */

/**
  * @}
  */

/** @defgroup OPA_LL_EC_BIAS_VOLTAGE OPA Bias Voltage
  * @{
  */
#define LL_OPA_BIASVOLTAGE_VCC_2    OPA_OCR_OPA_VBIAS_0           /*!< Bias Voltage 1/2VCC */
#define LL_OPA_BIASVOLTAGE_VCC_5    OPA_OCR_OPA_VBIAS_1           /*!< Bias Voltage 1/5VCC */
/**
  * @}
  */

/** @defgroup OPA_LL_EC_ADC_TRIGGER_SELECT OPA ADC Trigger Select
  * @{
  */
#define LL_OPA_ADC_TRIGGER_SELECT_EOSMP    0x00000000                                               /*!< ADC EOSMP Trigger */
#define LL_OPA_ADC_TRIGGER_SELECT_EOC      OPA_OCR_OPA_ADC_TRG_SEL_0                                /*!< ADC EOC Trigger */
#define LL_OPA_ADC_TRIGGER_SELECT_EOSEQ    OPA_OCR_OPA_ADC_TRG_SEL_1                                /*!< ADC EOSEQ Trigger */
#define LL_OPA_ADC_TRIGGER_SELECT_EOH      (OPA_OCR_OPA_ADC_TRG_SEL_0 | OPA_OCR_OPA_ADC_TRG_SEL_1)  /*!< ADC EOH Trigger */
/**
  * @}
  */
 
/**
  * @}
  */
/* Exported functions --------------------------------------------------------*/
/** @defgroup OPA_LL_Exported_Functions OPA Exported Functions
  * @{
  */

/** @defgroup OPA_LL_EF_CONFIGURATION_OPA_INSTANCE Configuration of OPA hierarchical scope: OPA instance
  * @{
  */

/**
  * @brief  Set OPA functional mode by setting internal connections.
  *         OPA operation in standalone, follower, ...
  * @rmtoll CSR      OPA_VMSEL          LL_OPA_SetFunctionalMode
  * @param  OPAx OPA instance
  * @param  FunctionalMode This parameter can be one of the following values:
  *         @arg @ref LL_OPA_MODE_STANDALONE
  *         @arg @ref LL_OPA_MODE_PGA
  *         @arg @ref LL_OPA_MODE_FOLLOWER
  *         @arg @ref LL_OPA_MODE_PGA_IO0_BIAS
  * @retval None
  */
__STATIC_INLINE void LL_OPA_SetFunctionalMode(OPA_TypeDef *OPAx, uint32_t FunctionalMode)
{
  MODIFY_REG(OPAx->OCR, OPA_OCR_OPA_VMSEL | OPA_OCR_OPA_PGA_GAIN_3, FunctionalMode);
}

/**
  * @brief  Get OPA functional mode from setting of internal connections.
  *         OPA operation in standalone, follower, ...
  * @rmtoll OCR      OPA_VMSEL          LL_OPA_GetFunctionalMode
  * @param  OPAx OPA instance
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_OPA_MODE_STANDALONE
  *         @arg @ref LL_OPA_MODE_PGA
  *         @arg @ref LL_OPA_MODE_FOLLOWER
  *         @arg @ref LL_OPA_MODE_PGA_IO0_BIAS
  */
__STATIC_INLINE uint32_t LL_OPA_GetFunctionalMode(OPA_TypeDef *OPAx)
{
  return (uint32_t)(READ_BIT(OPAx->OCR, OPA_OCR_OPA_VMSEL | OPA_OCR_OPA_PGA_GAIN_3));
}

/**
  * @brief  Set OPA PGA gain.
  * @note   Preliminarily, OPA must be set in mode PGA
  *         using function @ref LL_OPA_SetFunctionalMode().
  * @rmtoll OCR      OPA_PGA_GAIN         LL_OPA_SetPGAGain
  * @param  OPAx OPA instance
  * @param  PGAGain This parameter can be one of the following values:
  *         @arg @ref LL_OPA_PGA_GAIN_2_OR_MINUS_1
  *         @arg @ref LL_OPA_PGA_GAIN_4_OR_MINUS_3
  *         @arg @ref LL_OPA_PGA_GAIN_8_OR_MINUS_7
  *         @arg @ref LL_OPA_PGA_GAIN_16_OR_MINUS_15
  *         @arg @ref LL_OPA_PGA_GAIN_32_OR_MINUS_31
  * @retval None
  */
__STATIC_INLINE void LL_OPA_SetPGAGain(OPA_TypeDef *OPAx, uint32_t PGAGain)
{
  MODIFY_REG(OPAx->OCR,OPA_OCR_OPA_PGA_GAIN_0 | OPA_OCR_OPA_PGA_GAIN_1 | OPA_OCR_OPA_PGA_GAIN_2, PGAGain);
}

/**
  * @brief  Get OPA PGA gain.
  * @note   Preliminarily, OPA must be set in mode PGA
  *         using function @ref LL_OPA_SetFunctionalMode().
  * @rmtoll OCR      OPA_PGA_GAIN         LL_OPA_GetPGAGain
  * @param  OPAx OPA instance
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_OPA_PGA_GAIN_2_OR_MINUS_1
  *         @arg @ref LL_OPA_PGA_GAIN_4_OR_MINUS_3
  *         @arg @ref LL_OPA_PGA_GAIN_8_OR_MINUS_7
  *         @arg @ref LL_OPA_PGA_GAIN_16_OR_MINUS_15
  *         @arg @ref LL_OPA_PGA_GAIN_32_OR_MINUS_31
  */
__STATIC_INLINE uint32_t LL_OPA_GetPGAGain(OPA_TypeDef *OPAx)
{
  return (uint32_t)(READ_BIT(OPAx->OCR, OPA_OCR_OPA_PGA_GAIN_0 | OPA_OCR_OPA_PGA_GAIN_1 | OPA_OCR_OPA_PGA_GAIN_2));
}

/**
  * @brief  Enable the OPA.
  * @rmtoll CR      OPAEN            LL_OPA_Enable
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_Enable(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->CR, OPA_CR_OPAEN);
}

/**
  * @brief  Disable the OPA.
  * @rmtoll CR      OPAEN            LL_OPA_Disable
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_Disable(OPA_TypeDef *OPAx)
{
  CLEAR_BIT(OPAx->CR, OPA_CR_OPAEN);
}

/**
  * @brief  Get the OPA Enable state.
  * @rmtoll CR      OPAEN            LL_OPA_IsEnabled
  * @param  OPAx OPA instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_OPA_IsEnabled(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->CR, OPA_CR_OPAEN) == (OPA_CR_OPAEN)) ? 1UL : 0UL);
}

/**
  * @brief  Enable OPA output connected to IO.
  * @rmtoll OCR      OPA_EXTOEN     LL_OPA_EnableExternalOutput
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_EnableExternalOutput(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->OCR, OPA_OCR_OPA_EXTOEN);
}

/**
  * @brief  Disable OPA output connected to IO.
  * @rmtoll OCR      OPA_EXTOEN     LL_OPA_DisableExternalOutput
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_DisableExternalOutput(OPA_TypeDef *OPAx)
{
  CLEAR_BIT(OPAx->OCR, OPA_OCR_OPA_EXTOEN);
}

/**
  * @brief  Check if OPA output connected to IO is enabled
  * @rmtoll OCR      OPA_EXTOEN     LL_OPA_IsEnabledExternaldOutput
  * @param  OPAx OPA instance
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t LL_OPA_IsEnabledExternaldOutput(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->OCR, OPA_OCR_OPA_EXTOEN) == (OPA_OCR_OPA_EXTOEN)) ? 1UL : 0UL);
}

/**
  * @brief  Set OPA ADC Trigger Select.
  * @rmtoll OCR      OPA_ADC_TRG_SEL         LL_OPA_SetADCTriggerSelect
  * @param  OPAx OPA instance
  * @param  TriggerSelect This parameter can be one of the following values:
  *         @arg @ref LL_OPA_ADC_TRIGGER_SELECT_EOSMP
  *         @arg @ref LL_OPA_ADC_TRIGGER_SELECT_EOC
  *         @arg @ref LL_OPA_ADC_TRIGGER_SELECT_EOSEQ
  *         @arg @ref LL_OPA_ADC_TRIGGER_SELECT_EOH
  * @retval None
  */
__STATIC_INLINE void LL_OPA_SetADCTriggerSelect(OPA_TypeDef *OPAx, uint32_t TriggerSelect)
{
  MODIFY_REG(OPAx->OCR,OPA_OCR_OPA_ADC_TRG_SEL, TriggerSelect);
}

/**
  * @brief  Get OPA ADC Trigger Select.
  * @rmtoll OCR      OPA_ADC_TRG_SEL         LL_OPA_GetADCTriggerSelect
  * @param  OPAx OPA instance
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_OPA_ADC_TRIGGER_SELECT_EOSMP
  *         @arg @ref LL_OPA_ADC_TRIGGER_SELECT_EOC
  *         @arg @ref LL_OPA_ADC_TRIGGER_SELECT_EOSEQ
  *         @arg @ref LL_OPA_ADC_TRIGGER_SELECT_EOH
  */
__STATIC_INLINE uint32_t LL_OPA_GetADCTriggerSelect(OPA_TypeDef *OPAx)
{
  return (uint32_t)(READ_BIT(OPA1->OCR, OPA_OCR_OPA_ADC_TRG_SEL));
}

/**
  * @brief  Enable OPA Swap
  * @rmtoll OCR      OPA_SWAP     LL_OPA_EnableSwap
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_EnableSwap(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->OCR, OPA_OCR_OPA_SWAP);
}

/**
  * @brief  Check if OPA Swap enabled
  * @rmtoll OCR      OPA_SWAP     LL_OPA_IsEnabledSwap
  * @param  OPAx OPA instance
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t LL_OPA_IsEnabledSwap(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->OCR, OPA_OCR_OPA_SWAP) == (OPA_OCR_OPA_SWAP)) ? 1UL : 0UL);
}


/**
  * @brief  Enable OPA ADC Trigger Swap
  * @rmtoll OCR      OPA_ADC_TRG_SEL     LL_OPA_EnableADCTriggerSwap
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_EnableADCTriggerSwap(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->OCR, OPA_OCR_OPA_ADC_SWAP_EN);
}

/**
  * @brief  Disable OPA ADC Trigger Swap
  * @rmtoll OCR      OPA_ADC_TRG_SEL     LL_OPA_DisableADCTriggerSwap
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_DisableADCTriggerSwap(OPA_TypeDef *OPAx)
{
  CLEAR_BIT(OPAx->OCR, OPA_OCR_OPA_ADC_SWAP_EN);
}

/**
  * @brief  Check if OPA ADC Trigger Swap enabled
  * @rmtoll OCR      OPA_ADC_TRG_SEL     LL_OPA_IsEnabledADCTriggerSwap
  * @param  OPAx OPA instance
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t LL_OPA_IsEnabledADCTriggerSwap(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->OCR, OPA_OCR_OPA_ADC_SWAP_EN) == (OPA_OCR_OPA_ADC_SWAP_EN)) ? 1UL : 0UL);
}

/**
  * @brief  Enable OPA Bias
  * @rmtoll OCR      OPA_VBSEL     LL_OPA_EnableBiasSelection
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_EnableBiasSelection(OPA_TypeDef *OPAx)
{
  SET_BIT(OPAx->OCR, OPA_OCR_OPA_VBSEL);
}

/**
  * @brief  Disable OPA Bias
  * @rmtoll OCR      OPA_VBSEL     LL_OPA_DisableBiasSelection
  * @param  OPAx OPA instance
  * @retval None
  */
__STATIC_INLINE void LL_OPA_DisableBiasSelection(OPA_TypeDef *OPAx)
{
  CLEAR_BIT(OPAx->OCR, OPA_OCR_OPA_VBSEL);
}

/**
  * @brief  Check if OPA Bias enabled
  * @rmtoll OCR      OPA_VBSEL     LL_OPA_IsEnabledBiasSelection
  * @param  OPAx OPA instance
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t LL_OPA_IsEnabledBiasSelection(OPA_TypeDef *OPAx)
{
  return ((READ_BIT(OPAx->OCR, OPA_OCR_OPA_VBSEL) == (OPA_OCR_OPA_VBSEL)) ? 1UL : 0UL);
}



/**
  * @brief  Set OPA Bias Voltage.
  * @rmtoll OCR      OPA_VBIAS         LL_OPA_SetBiasVoltage
  * @param  OPAx OPA instance
  * @param  BiasVoltage This parameter can be one of the following values:
  *         @arg @ref LL_OPA_BIASVOLTAGE_VCC_2
  *         @arg @ref LL_OPA_BIASVOLTAGE_VCC_5
  * @retval None
  */
__STATIC_INLINE void LL_OPA_SetBiasVoltage(OPA_TypeDef *OPAx, uint32_t BiasVoltage)
{
  MODIFY_REG(OPAx->OCR,OPA_OCR_OPA_VBIAS, BiasVoltage);
}

/**
  * @brief  Get OPA Bias Voltage.
  * @rmtoll OCR      OPA_VBIAS         LL_OPA_GetBiasVoltage
  * @param  OPAx OPA instance
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_OPA_BIASVOLTAGE_VCC_2
  *         @arg @ref LL_OPA_BIASVOLTAGE_VCC_5
  */
__STATIC_INLINE uint32_t LL_OPA_GetBiasVoltage(OPA_TypeDef *OPAx)
{
  return (uint32_t)(READ_BIT(OPAx->OCR, OPA_OCR_OPA_VBIAS));
}

/**
  * @}
  */


#if defined(USE_FULL_LL_DRIVER)
/** @defgroup OPA_LL_EF_Init Initialization and de-initialization functions
  * @{
  */
ErrorStatus LL_OPA_DeInit(OPA_TypeDef *OPAx);
/**
  * @}
  */
#endif /* USE_FULL_LL_DRIVER */

/**
  * @}
  */

/**
  * @}
  */

#endif /* OPA */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __PY32F032_LL_OPA_H */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
